<?php
defined('ABSPATH') or die;

class SiteDataProduct
{

    public static $product;
    public static $siteProductId = 0;

    /**
     * Get site product object
     *
     * @param int $product_id
     *
     * @return object $product
     */
    public static function getProduct($product_id)
    {
        return $post = get_post($product_id);
    }

    /**
     * Get full data product for np
     *
     * @param bool $editor
     *
     * @return array $product_data
     */
    public static function getProductData($editor = false)
    {
        $product_data = array(
            'product'               => self::$product,
            'type'                  => 'simple',
            'title'                 => self::getProductTitle(),
            'desc'                  => self::getProductDesc(),
            'image_url'             => self::getProductImageUrl(),
            'price'                 => self::getProductPrice($editor),
            'price_old'             => self::getProductPriceOld($editor),
            'add_to_cart_text'      => 'Add to cart',
            'attributes'            => array(),
            'variations_attributes' => array(),
            'gallery_images'        => self::getProductImagesUrls(),
            'tabs'                  => array(),
            'meta'                  => '',
            'categories'            => self::getProductCategories(),
            'product-is-new'        => self::getProductIsNew(),
            'product-sale'          => self::getProductSale(),
            'product-out-of-stock'  => self::getProductOutOfStock(),
        );
        return $product_data;
    }

    /**
     * Get product title
     *
     * @return string $title
     */
    public static function getProductTitle()
    {
        return $title = isset(self::$product->post_title) ? self::$product->post_title : 'Product title';
    }

    /**
     * Get product description
     *
     * @return string $desc
     */
    public static function getProductDesc()
    {
        $desc = isset(self::$product->post_content) ? self::$product->post_content : 'Product description';
        return plugin_trim_long_str($desc, 150);
    }

    /**
     * Get product image url
     *
     * @return string $image_url
     */
    public static function getProductImageUrl()
    {
        $image = isset(self::$product->image) ? self::$product->image : '';
        $image_url = isset($image['url']) ? np_data_provider()->fixImagePaths($image['url']) : '';
        return $image_url;
    }

    /**
     * Get product price
     *
     * @param bool $editor
     *
     * @return int $price
     */
    public static function getProductPrice($editor = false)
    {
        $price = isset(self::$product->price) ? self::$product->price : '';
        $price = str_replace('$', '_dollar_symbol_', $price);
        return $price;
    }

    /**
     * Get product price old
     *
     * @param bool $editor
     *
     * @return int $price_old
     */
    public static function getProductPriceOld($editor = false)
    {
        $price_old = isset(self::$product->priceOld) ? self::$product->priceOld : '';
        $price_old = str_replace('$', '_dollar_symbol_', $price_old);
        return $price_old;
    }

    /**
     * Get product gallery images
     *
     * @return array $images urls
     */
    public static function getProductImagesUrls()
    {
        $images = isset(self::$product->gallery_images) && self::$product->gallery_images ? self::$product->gallery_images : array();
        $imagesUrls = array();
        foreach ($images as $image) {
            if ($image['url']) {
                $imagesUrls[] = np_data_provider()->fixImagePaths($image['url']);
            }
        }
        return $imagesUrls;
    }

    /**
     * Get button add to cart html
     *
     * @param string $button_html
     * @param array  $controlOptions
     * @param array  $allProducts
     *
     * @return string $button_html
     */
    public static function getProductButtonHtml($button_html, $controlOptions, $allProducts)
    {
        $addToCart = false;
        $goToProduct = false;
        if (isset($controlOptions['clickType'])) {
            if ($controlOptions['clickType'] === 'go-to-page') {
                $goToProduct = true;
            } else {
                $addToCart = true;
            }
        }
        if (self::$product && self::$siteProductId) {
            $button_html = str_replace('data-product-id=""', 'data-product-id="' . self::$siteProductId  . '"', $button_html);
            $allCategories = get_option('site_products_categories') ? json_decode(get_option('site_products_categories'), true) : array();
            $productInfo = $allProducts[self::$siteProductId];
            $productCategories = isset($productInfo['categories']) ? $productInfo['categories'] : array();
            $productInfo['categoriesData'] = NpDataProvider::getCategoriesData($productCategories, $allCategories);
            $button_html = str_replace('<a', '<a data-product="' . htmlspecialchars(json_encode($productInfo))  . '"', $button_html);
        }
        if (!$allProducts) {
            if ($goToProduct) {
                return preg_replace_callback(
                    '/href=[\"\']{1}product-?(\d+)[\"\']{1}/',
                    function ($hrefMatch) {
                        return 'href="' . home_url('?productId=' . $hrefMatch[1]) . '"';
                    },
                    $button_html
                );
            }
        }
        if ($addToCart) {
            $button_html = str_replace('class="', 'class="u-add-to-cart-link ', $button_html);
            $button_html = str_replace('u-dialog-link', '', $button_html);
        }
        return $button_html;
    }

    /**
     * Get product categories
     *
     * @return array $categories
     */
    public static function getProductCategories() {
        $categories = array(
            0 => array(
                'id' => 0,
                'title' => 'Uncategorized',
                'link' => '#',
            )
        );
        $all_categories = get_option('site_products_categories') ? json_decode(get_option('site_products_categories'), true) : array();
        $product_categories = isset(self::$product->categories) ? self::$product->categories : array();
        if ($product_categories) {
            $categories = array();
            foreach ($product_categories as $id) {
                $category = self::findElementById($all_categories, $id);
                if ($category) {
                    array_push(
                        $categories,
                        array(
                            'id'    => $id,
                            'title' => isset($category['title']) ? $category['title'] : 'Uncategorized',
                            'link'  => home_url('?productsList&categoryId=' . $id),
                        )
                    );
                }
            }
        }
        return $categories;
    }

    /**
     * Find category by id
     *
     * @param array $categories
     * @param int   $cat_id
     *
     * @return mixed|null
     */
    public static function findElementById($categories, $cat_id) {
        foreach ($categories as $element) {
            if ($element['id'] == $cat_id) {
                return $element;
            }
        }
        return null;
    }

    /**
     * Product is new
     */
    public static function getProductIsNew() {
        $currentDate = (int) (microtime(true) * 1000);
        if (isset(self::$product->post_date)) {
            $createdDate = (int) strtotime(self::$product->post_date) * 1000;
        } else {
            $createdDate = $currentDate;
        }
        $milliseconds30Days = 30 * (60 * 60 * 24 * 1000); // 30 days in milliseconds
        if (($currentDate - $createdDate) <= $milliseconds30Days) {
            return true;
        }
        return false;
    }

    /**
     * Sale for product
     */
    public static function getProductSale() {
        $price = 0;
        if (isset(self::$product->price)) {
            $price = self::extractNumber(self::$product->price);
        }
        $oldPrice = 0;
        if (isset(self::$product->priceOld)) {
            $oldPrice = self::extractNumber(self::$product->priceOld);
        }
        $sale = '';
        if ($price && $oldPrice && $price < $oldPrice) {
            $sale = '-' . (int) ( 100 - ( $price * 100 / $oldPrice ) ) . '%';
        }
        return $sale;
    }

    /**
     * Out of stock for product
     */
    public static function getProductOutOfStock() {
        return isset(self::$product->outOfStock) && self::$product->outOfStock === 'true' ? true : false;
    }

    /**
     * @param $string
     *
     * @return float|int
     */
    public static function extractNumber($string) {
        preg_match('/[\d\.]+/', $string, $matches);
        return isset($matches[0]) ? (float) $matches[0] : 0;
    }
}

/**
 * Construct SiteDataProduct object
 *
 * @param int  $product_id      Product Id from cms
 * @param int  $site_product_id Site Product Id
 * @param bool $editor          Need to check editor or live site
 *
 * @return array SiteDataProduct
 */
function site_data_product($product_id = 0, $site_product_id = 0, $editor = false)
{
    SiteDataProduct::$siteProductId = $site_product_id;
    SiteDataProduct::$product = SiteDataProduct::getProduct($product_id);
    return SiteDataProduct::$product ? SiteDataProduct::getProductData($editor) : array();
}

/**
 * Init Site products categories
 */
function custom_product_category_taxonomy() {
    $labels = array(
        'name' => 'Site products categories',
        'singular_name' => 'Site products category',
        'menu_name' => 'Site products categories'
    );
    $args = array(
        'labels' => $labels,
        'public' => true,
        'hierarchical' => true,
        'show_ui' => true,
        'show_admin_column' => true,
        'query_var' => true,
        'rewrite' => array( 'slug' => 'site_product_category' ),
    );
    register_taxonomy('site_product_category', 'site_product', $args);
}
add_action('init', 'custom_product_category_taxonomy');

/**
 * Add scripts and styles for woocommerce
 */
function add_np_shop_scripts() {
    global $post;
    $post_id = isset($post->ID) ? $post->ID : 0;
    if (np_data_provider($post_id)->isNp()) {
        wp_register_script('shop-np-scripts', APP_PLUGIN_URL . 'includes/shop/js/shop-np-scripts.js', array('jquery'), time());
        wp_enqueue_script('shop-np-scripts');
    }
}
add_action('wp_enqueue_scripts', 'add_np_shop_scripts', 1003);