<?php
/*
Plugin Name: Books Plugin
Plugin URI:  https://www.linkedin.com/in/sudeepa-das-590572225/
Description: A plugin to manage books.
Version:     1.0
Author:      Sudeepa Das
Author URI:  https://www.linkedin.com/in/sudeepa-das-590572225/
License:     GPL2
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}
//require_once plugin_dir_path(__FILE__) . 'includes/db-setup.php';

// Register Custom Post Type
function books_plugin_register_post_type() {
    $labels = array(
        'name'                  => _x('Books', 'Post Type General Name', 'text_domain'),
        'singular_name'         => _x('Book', 'Post Type Singular Name', 'text_domain'),
        'menu_name'             => __('Books', 'text_domain'),
        'name_admin_bar'        => __('Book', 'text_domain'),
        'archives'              => __('Book Archives', 'text_domain'),
        'attributes'            => __('Book Attributes', 'text_domain'),
        'parent_item_colon'     => __('Parent Book:', 'text_domain'),
        'all_items'             => __('All Books', 'text_domain'),
        'add_new_item'          => __('Add New Book', 'text_domain'),
        'add_new'               => __('Add New', 'text_domain'),
        'new_item'              => __('New Book', 'text_domain'),
        'edit_item'             => __('Edit Book', 'text_domain'),
        'update_item'           => __('Update Book', 'text_domain'),
        'view_item'             => __('View Book', 'text_domain'),
        'view_items'            => __('View Books', 'text_domain'),
        'search_items'          => __('Search Book', 'text_domain'),
        'not_found'             => __('Not found', 'text_domain'),
        'not_found_in_trash'    => __('Not found in Trash', 'text_domain'),
        'featured_image'        => __('Featured Image', 'text_domain'),
        'set_featured_image'    => __('Set featured image', 'text_domain'),
        'remove_featured_image' => __('Remove featured image', 'text_domain'),
        'use_featured_image'    => __('Use as featured image', 'text_domain'),
        'insert_into_item'      => __('Insert into book', 'text_domain'),
        'uploaded_to_this_item' => __('Uploaded to this book', 'text_domain'),
        'items_list'            => __('Books list', 'text_domain'),
        'items_list_navigation' => __('Books list navigation', 'text_domain'),
        'filter_items_list'     => __('Filter books list', 'text_domain'),
    );
    $args = array(
        'label'                 => __('Book', 'text_domain'),
        'description'           => __('A custom post type for books', 'text_domain'),
        'labels'                => $labels,
        'supports'              => array('title', 'editor', 'thumbnail', 'comments', 'revisions'),
        'taxonomies'            => array('genre', 'author', 'post_tag'),
        'hierarchical'          => false,
        'public'                => true,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'menu_position'         => 5,
        'show_in_admin_bar'     => true,
        'show_in_nav_menus'     => true,
        'can_export'            => true,
        'has_archive'           => true,
        'exclude_from_search'   => false,
        'publicly_queryable'    => true,
        'capability_type'       => 'post',
    );
    register_post_type('book', $args);
}
add_action('init', 'books_plugin_register_post_type', 0);

// Register Custom Taxonomy: Author
function books_plugin_register_taxonomy_author() {
    $labels = array(
        'name'                       => _x('Authors', 'Taxonomy General Name', 'text_domain'),
        'singular_name'              => _x('Author', 'Taxonomy Singular Name', 'text_domain'),
        'menu_name'                  => __('Authors', 'text_domain'),
        'all_items'                  => __('All Authors', 'text_domain'),
        'parent_item'                => __('Parent Author', 'text_domain'),
        'parent_item_colon'          => __('Parent Author:', 'text_domain'),
        'new_item_name'              => __('New Author Name', 'text_domain'),
        'add_new_item'               => __('Add New Author', 'text_domain'),
        'edit_item'                  => __('Edit Author', 'text_domain'),
        'update_item'                => __('Update Author', 'text_domain'),
        'view_item'                  => __('View Author', 'text_domain'),
        'separate_items_with_commas' => __('Separate authors with commas', 'text_domain'),
        'add_or_remove_items'        => __('Add or remove authors', 'text_domain'),
        'choose_from_most_used'      => __('Choose from the most used', 'text_domain'),
        'popular_items'              => __('Popular Authors', 'text_domain'),
        'search_items'               => __('Search Authors', 'text_domain'),
        'not_found'                  => __('Not Found', 'text_domain'),
        'no_terms'                   => __('No authors', 'text_domain'),
        'items_list'                 => __('Authors list', 'text_domain'),
        'items_list_navigation'      => __('Authors list navigation', 'text_domain'),
    );
    $args = array(
        'labels'                     => $labels,
        'hierarchical'               => true,
        'public'                     => true,
        'show_ui'                    => true,
        'show_admin_column'          => true,
        'show_in_nav_menus'          => true,
        'show_tagcloud'              => true,
    );
    register_taxonomy('author', array('book'), $args);
}
add_action('init', 'books_plugin_register_taxonomy_author', 0);

// Register Custom Taxonomy: Genre
function books_plugin_register_taxonomy_genre() {
    $labels = array(
        'name'                       => _x('Genres', 'Taxonomy General Name', 'text_domain'),
        'singular_name'              => _x('Genre', 'Taxonomy Singular Name', 'text_domain'),
        'menu_name'                  => __('Genres', 'text_domain'),
        'all_items'                  => __('All Genres', 'text_domain'),
        'parent_item'                => __('Parent Genre', 'text_domain'),
        'parent_item_colon'          => __('Parent Genre:', 'text_domain'),
        'new_item_name'              => __('New Genre Name', 'text_domain'),
        'add_new_item'               => __('Add New Genre', 'text_domain'),
        'edit_item'                  => __('Edit Genre', 'text_domain'),
        'update_item'                => __('Update Genre', 'text_domain'),
        'view_item'                  => __('View Genre', 'text_domain'),
        'separate_items_with_commas' => __('Separate genres with commas', 'text_domain'),
        'add_or_remove_items'        => __('Add or remove genres', 'text_domain'),
        'choose_from_most_used'      => __('Choose from the most used', 'text_domain'),
        'popular_items'              => __('Popular Genres', 'text_domain'),
        'search_items'               => __('Search Genres', 'text_domain'),
        'not_found'                  => __('Not Found', 'text_domain'),
        'no_terms'                   => __('No genres', 'text_domain'),
        'items_list'                 => __('Genres list', 'text_domain'),
        'items_list_navigation'      => __('Genres list navigation', 'text_domain'),
    );
    $args = array(
        'labels'                     => $labels,
        'hierarchical'               => true,
        'public'                     => true,
        'show_ui'                    => true,
        'show_admin_column'          => true,
        'show_in_nav_menus'          => true,
        'show_tagcloud'              => true,
    );
    register_taxonomy('genre', array('book'), $args);
}
add_action('init', 'books_plugin_register_taxonomy_genre', 0);

// Add meta boxes
function books_plugin_add_meta_box() {
    // Meta box for Book Available On
    add_meta_box(
        'book_available_on',
        __('Book Available On', 'text_domain'),
        'books_plugin_render_meta_box',
        'book',
        'normal', // Set to 'normal' priority
        'high' // Set to 'high' position
    );

    // Meta box for Upload PDF
    add_meta_box(
        'book_pdf_section',
        __('Upload PDF', 'text_domain'),
        'books_plugin_render_pdf_meta_box',
        'book',
        'normal', // Set to 'normal' priority
        'high' // Set to 'high' position
    );
    // Meta box for Price
    add_meta_box(
        'book_price_section',
        __('Book Price', 'text_domain'),
        'books_plugin_render_price_meta_box',
        'book',
        'normal', // Set to 'normal' priority
        'high' // Set to 'high' position
    );
}
add_action('add_meta_boxes', 'books_plugin_add_meta_box');

// Render meta box content for PDF upload
function books_plugin_render_pdf_meta_box($post) {
    $pdf_url = get_post_meta($post->ID, 'book_pdf', true);
    wp_nonce_field('book_pdf_nonce', 'book_pdf_nonce_field');
    ?>
    <p>
        <label for="book_pdf_url"><?php _e('Upload PDF (Chapter 1 or Popular Part)', 'text_domain'); ?></label>
        <input type="text" id="book_pdf_url" name="book_pdf_url" value="<?php echo esc_url($pdf_url); ?>" readonly style="width: 80%;">
        <button class="button books-plugin-upload-pdf-button"><?php _e('Upload PDF', 'text_domain'); ?></button>
        <div class="books-plugin-uploaded-pdf">
            <?php if ($pdf_url) : ?>
                <span><?php _e('Current PDF:', 'text_domain'); ?> <a href="<?php echo esc_url($pdf_url); ?>"><?php echo esc_html(basename($pdf_url)); ?></a></span>
            <?php endif; ?>
        </div>
    </p>
    <?php
}


// Render meta box content
function books_plugin_render_meta_box($post) {
    $stored_data = get_post_meta($post->ID, 'book_available_on', true);
    wp_nonce_field('book_available_on_nonce', 'book_available_on_nonce_field');

    if (!is_array($stored_data) || empty($stored_data)) {
        $stored_data = array( array('store_name' => '', 'url' => '', 'reviews' => '', 'book_url' => '') );
    }
    ?>
    <div class="book-available-on-fields">
        <?php foreach ($stored_data as $index => $store_data) : ?>
            <div class="book-available-on-field">
                <h4><?php printf(__('Store %d', 'text_domain'), $index + 1); ?></h4>
                <p>
                    <label for="book_available_on_store_name_<?php echo $index; ?>"><?php _e('Store Name:', 'text_domain'); ?></label>
                    <input type="text" id="book_available_on_store_name_<?php echo $index; ?>" name="book_available_on[<?php echo $index; ?>][store_name]" value="<?php echo esc_attr($store_data['store_name']); ?>">
                </p>
                <p>
                    <label for="book_available_on_url_<?php echo $index; ?>"><?php _e('Store URL:', 'text_domain'); ?></label>
                    <input type="url" id="book_available_on_url_<?php echo $index; ?>" name="book_available_on[<?php echo $index; ?>][url]" value="<?php echo esc_url($store_data['url']); ?>">
                </p>
                <p>
                    <label for="book_available_on_reviews_<?php echo $index; ?>"><?php _e('Reviews:', 'text_domain'); ?></label>
                    <input type="number" id="book_available_on_reviews_<?php echo $index; ?>" name="book_available_on[<?php echo $index; ?>][reviews]" min="0" max="5" step="0.5" value="<?php echo esc_attr($store_data['reviews']); ?>">
                </p>
                <p>
                    <label for="book_available_on_book_url_<?php echo $index; ?>"><?php _e('Book URL in Store:', 'text_domain'); ?></label>
                    <input type="url" id="book_available_on_book_url_<?php echo $index; ?>" name="book_available_on[<?php echo $index; ?>][book_url]" value="<?php echo esc_url($store_data['book_url']); ?>">
                </p>
            </div>
        <?php endforeach; ?>
    </div>
    <button type="button" class="button" id="add-new-store"><?php _e('Add New Store', 'text_domain'); ?></button>
    <script>
        jQuery(document).ready(function($) {
            $('#add-new-store').on('click', function() {
                var storeIndex = $('.book-available-on-field').length;
                var newStore = `
                <div class="book-available-on-field">
                    <h4><?php _e('New Store', 'text_domain'); ?></h4>
                    <p>
                        <label for="book_available_on_store_name_${storeIndex}"><?php _e('Store Name:', 'text_domain'); ?></label>
                        <input type="text" id="book_available_on_store_name_${storeIndex}" name="book_available_on[${storeIndex}][store_name]" value="">
                    </p>
                    <p>
                        <label for="book_available_on_url_${storeIndex}"><?php _e('Store URL:', 'text_domain'); ?></label>
                        <input type="url" id="book_available_on_url_${storeIndex}" name="book_available_on[${storeIndex}][url]" value="">
                    </p>
                    <p>
                        <label for="book_available_on_reviews_${storeIndex}"><?php _e('Reviews:', 'text_domain'); ?></label>
                        <input type="number" id="book_available_on_reviews_${storeIndex}" name="book_available_on[${storeIndex}][reviews]" min="0" max="5" step="0.5"></textarea>
                    </p>
                    <p>
                        <label for="book_available_on_book_url_${storeIndex}"><?php _e('Book URL in Store:', 'text_domain'); ?></label>
                        <input type="url" id="book_available_on_book_url_${storeIndex}" name="book_available_on[${storeIndex}][book_url]" value="">
                    </p>
                </div>
                `;
                $('.book-available-on-fields').append(newStore);
            });
        });
    </script>
    <?php
}
// Render meta box content for Price
function books_plugin_render_price_meta_box($post) {
    $price = get_post_meta($post->ID, 'book_price', true);
    wp_nonce_field('book_price_nonce', 'book_price_nonce_field');
    ?>
    <p>
        <label for="book_price"><?php _e('Price:', 'text_domain'); ?></label>
        <input type="number" id="book_price" name="book_price" value="<?php echo esc_attr($price); ?>" step="0.01" min="0" style="width: 80%;">
    </p>
    <?php
}

/// Save meta box data
function books_plugin_save_meta_box_data($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    // Verify nonce for Book Available On
    if (!isset($_POST['book_available_on_nonce_field']) || !wp_verify_nonce($_POST['book_available_on_nonce_field'], 'book_available_on_nonce')) return;

    // Verify nonce for PDF upload
    if (!isset($_POST['book_pdf_nonce_field']) || !wp_verify_nonce($_POST['book_pdf_nonce_field'], 'book_pdf_nonce')) return;

    // Verify nonce for Price
    if (!isset($_POST['book_price_nonce_field']) || !wp_verify_nonce($_POST['book_price_nonce_field'], 'book_price_nonce')) return;

    if (!current_user_can('edit_post', $post_id)) return;

    // Save Book Available On
    if (isset($_POST['book_available_on'])) {
        $data = $_POST['book_available_on'];

        // Sanitize and save data for Book Available On
        foreach ($data as $key => $value) {
            if (is_string($value)) {
                $data[$key] = sanitize_text_field($value);
            } elseif (is_array($value)) {
                $data[$key] = array_map('sanitize_text_field', $value);
            }
        }

        update_post_meta($post_id, 'book_available_on', $data);
    }

    // Handle PDF URL
    if (isset($_POST['book_pdf_url'])) {
        $pdf_url = esc_url_raw($_POST['book_pdf_url']);
        update_post_meta($post_id, 'book_pdf', $pdf_url);
    }

    // Handle Price
    if (isset($_POST['book_price'])) {
        $price = floatval($_POST['book_price']);
        update_post_meta($post_id, 'book_price', $price);
    }
}
add_action('save_post_book', 'books_plugin_save_meta_box_data');

function books_plugin_enqueue_media_uploader() {
    wp_enqueue_media();
    wp_enqueue_script('books-plugin-media-uploader', plugin_dir_url(__FILE__) . 'js/media-uploader.js', array('jquery'), null, true);
}
add_action('admin_enqueue_scripts', 'books_plugin_enqueue_media_uploader');