<?php
/*
Plugin Name: Tee-Time
Plugin URI: #
Description: Golf Tee-time booking solution!
Text Domain: teetime
Version: 1.0.0
Author: Sudeepa Das
Author URI: #
*/

// Include custom functions file from the include folder
include_once(plugin_dir_path(__FILE__) . 'includes/shortcode.php');

// Hook to register custom post type and taxonomy
add_action('init', 'teetime_register_golf_club_post_type');

function teetime_register_golf_club_post_type() {
    $labels = array(
        'name' => __('Golf Clubs', 'teetime'),
        'singular_name' => __('Golf Club', 'teetime'),
        'menu_name' => __('Golf Clubs', 'teetime'),
        'name_admin_bar' => __('Golf Club', 'teetime'),
        'add_new' => __('Add New', 'teetime'),
        'add_new_item' => __('Add New Golf Club', 'teetime'),
        'new_item' => __('New Golf Club', 'teetime'),
        'edit_item' => __('Edit Golf Club', 'teetime'),
        'view_item' => __('View Golf Club', 'teetime'),
        'all_items' => __('All Golf Clubs', 'teetime'),
        'search_items' => __('Search Golf Clubs', 'teetime'),
        'not_found' => __('No Golf Clubs found', 'teetime'),
    );

    $args = array(
        'labels' => $labels,
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'thumbnail', 'custom-fields'), // Removed 'price' from meta fields
        'menu_icon' => 'dashicons-calendar-alt',
        'show_in_rest' => true,
        'rewrite' => array('slug' => 'golf-clubs'),
    );

    register_post_type('golf-club', $args);

    // Register custom taxonomy for category
    teetime_register_golf_club_taxonomy();
}

// Function to register taxonomy for golf clubs
function teetime_register_golf_club_taxonomy() {
    $labels = array(
        'name' => __('Categories', 'teetime'),
        'singular_name' => __('Category', 'teetime'),
        'search_items' => __('Search Categories', 'teetime'),
        'all_items' => __('All Categories', 'teetime'),
        'parent_item' => __('Parent Category', 'teetime'),
        'parent_item_colon' => __('Parent Category:', 'teetime'),
        'edit_item' => __('Edit Category', 'teetime'),
        'update_item' => __('Update Category', 'teetime'),
        'add_new_item' => __('Add New Category', 'teetime'),
        'new_item_name' => __('New Category Name', 'teetime'),
        'menu_name' => __('Categories', 'teetime'),
    );

    $args = array(
        'hierarchical' => true,  // Like categories
        'labels' => $labels,
        'show_ui' => true,
        'show_admin_column' => true,
        'query_var' => true,
        'rewrite' => array('slug' => 'golf-club-category'),
        'show_in_rest' => true,
    );

    register_taxonomy('golf_club_category', array('golf-club'), $args);
}

// Add meta boxes for additional fields (without price)
add_action('add_meta_boxes', 'teetime_add_custom_meta_boxes');

function teetime_add_custom_meta_boxes() {
    add_meta_box(
        'golf_club_details',
        __('Golf Club Details', 'teetime'),
        'teetime_golf_club_meta_box_callback',
        'golf-club',
        'normal',
        'default'
    );
}

function teetime_golf_club_meta_box_callback($post) {
    // Add nonce for security
    wp_nonce_field('teetime_save_golf_club_data', 'teetime_golf_club_meta_box_nonce');

    $start_time = get_post_meta($post->ID, '_golf_club_start_time', true);
    $end_time = get_post_meta($post->ID, '_golf_club_end_time', true);
    $duration = get_post_meta($post->ID, '_golf_club_duration', true);
    $capacity = get_post_meta($post->ID, '_golf_club_capacity', true);

    ?>
    <p>
        <label for="golf_club_start_time"><?php _e('Start Time:', 'teetime'); ?></label>
        <input type="time" id="golf_club_start_time" name="golf_club_start_time" value="<?php echo esc_attr($start_time); ?>" class="widefat" />
    </p>
    <p>
        <label for="golf_club_end_time"><?php _e('End Time:', 'teetime'); ?></label>
        <input type="time" id="golf_club_end_time" name="golf_club_end_time" value="<?php echo esc_attr($end_time); ?>" class="widefat" />
    </p>
    <p>
        <label for="golf_club_duration"><?php _e('Duration (Minutes):', 'teetime'); ?></label>
        <input type="number" id="golf_club_duration" name="golf_club_duration" value="<?php echo esc_attr($duration); ?>" class="widefat" />
    </p>
    <p>
        <label for="golf_club_capacity"><?php _e('Capacity Per Slot:', 'teetime'); ?></label>
        <input type="number" id="golf_club_capacity" name="golf_club_capacity" value="<?php echo esc_attr($capacity); ?>" class="widefat" />
    </p>
    <?php
}

// Save meta box data (without price field)
add_action('save_post', 'teetime_save_golf_club_data');

function teetime_save_golf_club_data($post_id) {
    if (!isset($_POST['teetime_golf_club_meta_box_nonce']) || !wp_verify_nonce($_POST['teetime_golf_club_meta_box_nonce'], 'teetime_save_golf_club_data')) {
        return;
    }

    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    if (isset($_POST['post_type']) && 'golf-club' === $_POST['post_type']) {
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
    }

    $fields = [
        'golf_club_start_time' => '_golf_club_start_time',
        'golf_club_end_time' => '_golf_club_end_time',
        'golf_club_duration' => '_golf_club_duration',
        'golf_club_capacity' => '_golf_club_capacity',
    ];

    foreach ($fields as $field_name => $meta_key) {
        if (isset($_POST[$field_name])) {
            update_post_meta($post_id, $meta_key, sanitize_text_field($_POST[$field_name]));
        }
    }
}

// Enqueue jQuery and SweetAlert scripts
function enqueue_jquery() {
    wp_enqueue_script('jquery');
}
add_action('wp_enqueue_scripts', 'enqueue_jquery');

function enqueue_sweetalert() {
    wp_enqueue_script('sweetalert-js', 'https://cdn.jsdelivr.net/npm/sweetalert2@11', array('jquery'), null, true);
}
add_action('wp_enqueue_scripts', 'enqueue_sweetalert');
// Hook to add submenu under Golf Clubs
add_action('admin_menu', 'teetime_add_bookings_submenu');

function teetime_add_bookings_submenu() {
    add_submenu_page(
        'edit.php?post_type=golf-club', // Parent slug (Golf Clubs menu)
        __('Bookings', 'teetime'), // Page title
        __('Bookings', 'teetime'), // Menu title
        'manage_options', // Capability
        'teetime-bookings', // Menu slug
        'teetime_bookings_page_callback' // Callback function
    );
}
function teetime_bookings_page_callback() {
    global $wpdb;

    // Pagination variables
    $items_per_page = 10;  // Number of bookings per page
    $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1; // Get current page
    $offset = ($current_page - 1) * $items_per_page; // Calculate offset for SQL query

    // Count the total number of bookings for pagination
    $total_items = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}teetime_booking");

    // Corrected SQL query to fetch bookings and payments data, with LIMIT and OFFSET for pagination
    $query = "
        SELECT b.*, p.payment_type, p.amount, ps.post_title
        FROM {$wpdb->prefix}teetime_booking AS b
        LEFT JOIN {$wpdb->prefix}teetime_payment AS p ON b.Id = p.booking_id
        LEFT JOIN {$wpdb->prefix}posts AS ps ON b.Gold_Course_id = ps.ID
        LIMIT $items_per_page OFFSET $offset
    ";

    // Execute the query
    $bookings = $wpdb->get_results($query);

    // Custom styles
    echo '<style>/* Custom styles for TeeTime Bookings table */
    .wrap h1 {
        font-size: 24px;
        margin-bottom: 20px;
    }

    .widefat.fixed {
        width: 100%;
        border-collapse: collapse;
        margin-bottom: 20px;
    }

    .widefat.fixed th {
        background-color: #f1f1f1;
        padding: 10px;
        text-align: left;
    }

    .widefat.fixed td {
        padding: 10px;
        border-bottom: 1px solid #eee;
    }

    .widefat.fixed tr:nth-child(even) {
        background-color: #f9f9f9;
    }

    .widefat.fixed tr:hover {
        background-color: #f1f1f1;
    }

    .widefat.fixed th, .widefat.fixed td {
        font-size: 14px;
    }

    /* Custom pagination styles */
    .tablenav-pages {
        display: flex;
        justify-content: center;
        align-items: center;
        margin-top: 20px;
    }

    .tablenav-pages a, .tablenav-pages span {
        padding: 5px 10px;
        margin: 0 2px;
        border: 1px solid #ddd;
        background-color: #f9f9f9;
        color: #0073aa;
        text-decoration: none;
    }

    .tablenav-pages a:hover {
        background-color: #0073aa;
        color: white;
    }

    .tablenav-pages .current {
        background-color: #0073aa;
        color: white;
        font-weight: bold;
    }

    .tablenav-pages .disabled {
        opacity: 0.6;
        pointer-events: none;
    }
    </style>';

    // Display the table
    echo '<div class="wrap">';
    echo '<h1>' . __('Bookings', 'teetime') . '</h1>';
    echo '<table class="widefat fixed" cellspacing="0">';
    echo '<thead><tr>';
    echo '<th>' . __('Name', 'teetime') . '</th>';
    echo '<th>' . __('Email', 'teetime') . '</th>';
    echo '<th>' . __('Golf Course', 'teetime') . '</th>';
    echo '<th>' . __('Accompany', 'teetime') . '</th>';
    echo '<th>' . __('Phone', 'teetime') . '</th>';
    echo '<th>' . __('Date', 'teetime') . '</th>';
    echo '<th>' . __('Location', 'teetime') . '</th>';
    echo '<th>' . __('Timeslot', 'teetime') . '</th>';
    echo '<th>' . __('Payment Type', 'teetime') . '</th>';
    echo '<th>' . __('Payment Amount', 'teetime') . '</th>';
    echo '</tr></thead>';
    echo '<tbody>';

    // Loop through bookings and display them
    if ($bookings) {
        foreach ($bookings as $booking) {
            echo '<tr>';
            echo '<td>' . esc_html($booking->Name) . '</td>';
            echo '<td>' . esc_html($booking->Email) . '</td>';
            echo '<td>' . esc_html($booking->post_title) . '</td>'; // Golf Course name
            echo '<td>' . esc_html($booking->Accompany) . '</td>';
            echo '<td>' . esc_html($booking->Phone) . '</td>';
            echo '<td>' . esc_html($booking->Date) . '</td>';
            echo '<td>' . esc_html($booking->Location) . '</td>';
            echo '<td>' . esc_html($booking->timeslot) . '</td>';
            echo '<td>' . esc_html($booking->payment_type ? $booking->payment_type : __('N/A', 'teetime')) . '</td>';
            echo '<td>' . esc_html($booking->amount ? $booking->amount : __('N/A', 'teetime')) . '</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="10">' . __('No bookings found', 'teetime') . '</td></tr>';
    }

    echo '</tbody></table>';

    // Pagination links
    $pagination_args = array(
        'base' => add_query_arg('paged', '%#%'),
        'format' => '',
        'current' => $current_page,
        'total' => ceil($total_items / $items_per_page),
        'prev_text' => __('&laquo; Previous'),
        'next_text' => __('Next &raquo;'),
    );
    
    echo '<div class="tablenav"><div class="tablenav-pages">';
    echo paginate_links($pagination_args);
    echo '</div></div>'; // .tablenav .tablenav-pages

    echo '</div>'; // .wrap
}
