import React, { useState } from 'react';
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";

// Define the interface for the accelerator data
interface AcceleratorItem {
  name: string;
  title: string;
  subtitle: string;
  solutionText: string;
  acceleratorIcon: string;
}

// Data structure for the different accelerator tabs
const acceleratorData: AcceleratorItem[] = [
  {
    name: "Portals",
    title: "Portals",
    subtitle: "Self-service customer and broker portals",
    solutionText: "View Solution",
    acceleratorIcon: "P", 
  },
  {
    name: "Claims",
    title: "Claims Automation",
    subtitle: "Automated claims processing and FNOL",
    solutionText: "Explore Claims",
    acceleratorIcon: "C",
  },
  {
    name: "Underwriting",
    title: "Underwriting Accelerator",
    subtitle: "AI-powered risk assessment and policy issuance",
    solutionText: "See Underwriting",
    acceleratorIcon: "U",
  },
  {
    name: "Policy servicing",
    title: "Policy Servicing",
    subtitle: "Seamless policy amendments and renewals",
    solutionText: "Details Here",
    acceleratorIcon: "PS",
  },
  {
    name: "Fraud analytics",
    title: "Fraud Analytics",
    subtitle: "Real-time fraud detection and prevention",
    solutionText: "Analyze Fraud",
    acceleratorIcon: "FA",
  },
  {
    name: "Reports",
    title: "Advanced Reports",
    subtitle: "Business intelligence and regulatory reporting",
    solutionText: "Generate Reports",
    acceleratorIcon: "R",
  },
];

export default function InsuranceAccelerators(): JSX.Element {
  // Set the initial active tab to the first item ("Portals")
  const [activeAccelerator, setActiveAccelerator] = useState<AcceleratorItem>(acceleratorData[0]);

  return (
    <section className="py-16 md:py-24">
      <div className="container mx-auto px-4 md:px-8 text-center">
        {/* Main Title and Subtitle */}
        <h2 className="text-3xl md:text-4xl lg:text-5xl font-bold tracking-tight mb-4">
          Pre-built insurance accelerators
        </h2>
        <p className="text-lg md:text-xl text-muted-foreground max-w-4xl mx-auto mb-12">
          Jumpstart your automation journey with building blocks tailored for the insurance
          industry, enabling faster enterprise adoption of intelligent automation technology.
        </p>

        {/* Tab Buttons */}
        <div className="flex flex-wrap justify-center gap-3 mb-12">
          {acceleratorData.map((item) => (
            <Button
              key={item.name}
              // The variant prop needs to be typed correctly if not inferred from the Button component definition
              // Assuming 'default' and 'outline' are valid types for the Button component
              variant={item.name === activeAccelerator.name ? "default" : "outline"}
              onClick={() => setActiveAccelerator(item)}
              className="rounded-full px-5 py-2 transition-colors duration-200"
            >
              {item.name}
            </Button>
          ))}
        </div>

        {/* Content Card (Box) */}
        <Card className="p-8 text-left bg-gray-50 border-gray-200 shadow-xl max-w-5xl mx-auto">
          <CardContent className="p-0 grid md:grid-cols-2 gap-8">
            {/* Text Content Area */}
            <div className="space-y-6">
              <h3 className="text-3xl font-bold">{activeAccelerator.title}</h3>
              <p className="text-lg text-muted-foreground">
                {activeAccelerator.subtitle}
              </p>
              <Button className="bg-primary hover:bg-primary/90 text-white rounded-md px-6 py-3">
                {activeAccelerator.solutionText}
              </Button>
            </div>

            {/* Visual/Image Placeholder Area */}
            <div className="flex items-center justify-center bg-white rounded-lg p-10 border border-gray-100 shadow-inner">
              <div className="text-center">
                <div className="w-20 h-20 rounded-full bg-blue-100 flex items-center justify-center mx-auto mb-3">
                  <span className="text-4xl font-semibold text-primary">
                    {activeAccelerator.acceleratorIcon}
                  </span>
                </div>
                <p className="text-sm font-medium text-muted-foreground">
                  {activeAccelerator.name} Accelerator
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </section>
  );
}