<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://profiles.wordpress.org/shoaib88/
 * @since      2.4
 *
 * @package    WP_Post_Comment_Rating
 * @subpackage WP_Post_Comment_Rating/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      2.4
 * @package    WP_Post_Comment_Rating
 * @subpackage WP_Post_Comment_Rating/includes
 * @author     Shoaib Saleem <shoaibsaleem20@gmail.com>
 */
class WP_Post_Comment_Rating {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    2.4
	 * @access   protected
	 * @var      WP_Post_Comment_Rating_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    2.4
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    2.4
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    2.4
	 */
	public function __construct() {
		if ( defined( 'WP_Post_Comment_Rating_VERSION' ) ) {
			$this->version = WP_Post_Comment_Rating_VERSION;
		} else {
			$this->version = '2.4';
		}
		$this->plugin_name = 'wp-post-comment-rating';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - WP_Post_Comment_Rating_Loader. Orchestrates the hooks of the plugin.
	 * - WP_Post_Comment_Rating_i18n. Defines internationalization functionality.
	 * - WP_Post_Comment_Rating_Admin. Defines all hooks for the admin area.
	 * - WP_Post_Comment_Rating_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    2.4
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wp-post-comment-rating-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wp-post-comment-rating-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-wp-post-comment-rating-admin.php';
		
		// Common methods and settings for public
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'common/class-wp-post-comment-rating-common.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'common/class-wp-post-comment-rating-settings.php';
		
		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wp-post-comment-rating-public.php';

		$this->loader = new WP_Post_Comment_Rating_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the WP_Post_Comment_Rating_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    2.4
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new WP_Post_Comment_Rating_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    2.4
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new WP_Post_Comment_Rating_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'wpcr_register_options');
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'wpcr_admin_links');
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    2.4
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new WP_Post_Comment_Rating_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
		$this->loader->add_action( 'comment_form_top', $plugin_public, 'wpcr_form_rate_field');
		$this->loader->add_action( 'comment_post', $plugin_public, 'wpcr_meta_data_save' );
		$this->loader->add_filter( 'preprocess_comment', $plugin_public, 'wpcr_meta_data_verify' );
		$this->loader->add_filter( 'the_tags', $plugin_public, 'wpcr_tag_aggr_val',10,3 );
		$this->loader->add_filter( 'comment_text', $plugin_public, 'wpcr_comment_text_vote_val',99,2);
		$this->loader->add_filter( 'the_content', $plugin_public, 'wpcr_show_nav_links_val',20, 1);
		$this->loader->add_action( 'wp_head', $plugin_public, 'wpcr_social_tags');
		$this->loader->add_action( 'wp_head', $plugin_public, 'wpcr_style_options_val'); 
		$this->loader->add_filter( 'the_content', $plugin_public, 'wpcr_rich_snippets_val', 30, 1);
		$this->loader->add_action( 'init', $plugin_public, 'wpcr_register_shortcodes' );
		   
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    2.4
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     2.4
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     2.4
	 * @return    WP_Post_Comment_Rating_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     2.4
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
