<?php
session_start();

$errors = isset($_SESSION['errors']) ? $_SESSION['errors'] : [];
$form_data = isset($_SESSION['form_data']) ? $_SESSION['form_data'] : [];
$success = isset($_GET['success']) ? true : false;

unset($_SESSION['errors']);
unset($_SESSION['form_data']);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Car Selection - Find Your Perfect Car</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }

        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
        }

        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
            font-weight: 700;
        }

        .header p {
            font-size: 16px;
            opacity: 0.9;
        }

        .form-container {
            padding: 40px 30px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 25px;
            font-size: 14px;
        }

        .alert-success {
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }

        .alert ul {
            margin-left: 20px;
            margin-top: 10px;
        }

        .form-group {
            margin-bottom: 25px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }

        .form-group label .required {
            color: #e74c3c;
        }

        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 15px;
            transition: all 0.3s ease;
            font-family: inherit;
        }

        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        textarea.form-control {
            resize: vertical;
            min-height: 100px;
        }

        .car-options {
            margin-top: 10px;
        }

        .car-option {
            display: flex;
            align-items: center;
            padding: 15px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            margin-bottom: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
            background: #fafafa;
        }

        .car-option:hover {
            border-color: #667eea;
            background: #f0f4ff;
            transform: translateX(5px);
        }

        .car-option input[type="checkbox"] {
            width: 20px;
            height: 20px;
            margin-right: 15px;
            cursor: pointer;
            accent-color: #667eea;
        }

        .car-option-content {
            flex: 1;
        }

        .car-option-title {
            font-weight: 600;
            color: #333;
            font-size: 16px;
            margin-bottom: 3px;
        }

        .car-option-desc {
            font-size: 13px;
            color: #666;
        }

        .car-option-icon {
            font-size: 30px;
            color: #667eea;
            margin-right: 15px;
        }

        .submit-btn {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 10px;
        }

        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
        }

        .submit-btn:active {
            transform: translateY(0);
        }

        .footer {
            text-align: center;
            padding: 20px;
            background: #f8f9fa;
            color: #666;
            font-size: 14px;
        }

        @media (max-width: 768px) {
            .container {
                margin: 0;
                border-radius: 0;
            }

            .header {
                padding: 30px 20px;
            }

            .header h1 {
                font-size: 26px;
            }

            .form-container {
                padding: 30px 20px;
            }

            .car-option {
                padding: 12px;
            }

            .car-option-icon {
                font-size: 25px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-car"></i> Car Selection Portal</h1>
            <p>Find your perfect car from our premium collection</p>
        </div>

        <div class="form-container">
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <strong><i class="fas fa-check-circle"></i> Success!</strong> Your response has been submitted successfully. We'll contact you soon!
                </div>
            <?php endif; ?>

            <?php if (!empty($errors)): ?>
                <div class="alert alert-error">
                    <strong><i class="fas fa-exclamation-circle"></i> Please fix the following errors:</strong>
                    <ul>
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo htmlspecialchars($error); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <form action="processform.php" method="POST" id="carForm">
                <div class="form-group">
                    <label for="name">Full Name <span class="required">*</span></label>
                    <input 
                        type="text" 
                        id="name" 
                        name="name" 
                        class="form-control" 
                        placeholder="Enter your full name"
                        value="<?php echo isset($form_data['name']) ? htmlspecialchars($form_data['name']) : ''; ?>"
                        required
                    >
                </div>

                <div class="form-group">
                    <label for="phone">Phone Number <span class="required">*</span></label>
                    <input 
                        type="tel" 
                        id="phone" 
                        name="phone" 
                        class="form-control" 
                        placeholder="Enter 10-digit phone number"
                        pattern="[0-9]{10}"
                        value="<?php echo isset($form_data['phone']) ? htmlspecialchars($form_data['phone']) : ''; ?>"
                        required
                    >
                </div>

                <div class="form-group">
                    <label for="email">Email Address <span class="required">*</span></label>
                    <input 
                        type="email" 
                        id="email" 
                        name="email" 
                        class="form-control" 
                        placeholder="Enter your email address"
                        value="<?php echo isset($form_data['email']) ? htmlspecialchars($form_data['email']) : ''; ?>"
                        required
                    >
                </div>

                <div class="form-group">
                    <label for="address">Address <span class="required">*</span></label>
                    <textarea 
                        id="address" 
                        name="address" 
                        class="form-control" 
                        placeholder="Enter your complete address"
                        required
                    ><?php echo isset($form_data['address']) ? htmlspecialchars($form_data['address']) : ''; ?></textarea>
                </div>

                <div class="form-group">
                    <label>Select Car Type(s) <span class="required">*</span></label>
                    <p style="font-size: 13px; color: #666; margin-bottom: 15px;">Choose one or more options</p>
                    
                    <div class="car-options">
                        <label class="car-option">
                            <input 
                                type="checkbox" 
                                name="car_types[]" 
                                value="Hatchback"
                                <?php echo (isset($form_data['car_types']) && in_array('Hatchback', $form_data['car_types'])) ? 'checked' : ''; ?>
                            >
                            <i class="fas fa-car car-option-icon"></i>
                            <div class="car-option-content">
                                <div class="car-option-title">Hatchback</div>
                                <div class="car-option-desc">Compact and fuel-efficient for city driving</div>
                            </div>
                        </label>

                        <label class="car-option">
                            <input 
                                type="checkbox" 
                                name="car_types[]" 
                                value="Sedan"
                                <?php echo (isset($form_data['car_types']) && in_array('Sedan', $form_data['car_types'])) ? 'checked' : ''; ?>
                            >
                            <i class="fas fa-car-side car-option-icon"></i>
                            <div class="car-option-content">
                                <div class="car-option-title">Sedan</div>
                                <div class="car-option-desc">Comfortable and spacious for family trips</div>
                            </div>
                        </label>

                        <label class="car-option">
                            <input 
                                type="checkbox" 
                                name="car_types[]" 
                                value="SUV"
                                <?php echo (isset($form_data['car_types']) && in_array('SUV', $form_data['car_types'])) ? 'checked' : ''; ?>
                            >
                            <i class="fas fa-truck-pickup car-option-icon"></i>
                            <div class="car-option-content">
                                <div class="car-option-title">SUV</div>
                                <div class="car-option-desc">Powerful and versatile for all terrains</div>
                            </div>
                        </label>
                    </div>
                </div>

                <button type="submit" class="submit-btn">
                    <i class="fas fa-paper-plane"></i> Submit Request
                </button>
            </form>
        </div>

        <div class="footer">
            <p>&copy; 2024 Car Selection Portal. All rights reserved.</p>
        </div>
    </div>

    <script>
       
        const carForm = document.getElementById('carForm');
        const nameInput = document.getElementById('name');
        const phoneInput = document.getElementById('phone');
        const emailInput = document.getElementById('email');

        phoneInput.addEventListener('input', function (e) {
            this.value = this.value.replace(/[^0-9]/g, '').slice(0, 10);
        });

        carForm.addEventListener('submit', async function (e) {
            e.preventDefault();

            const name = nameInput.value.trim();
            const phone = phoneInput.value.trim();
            const email = emailInput.value.trim();
            const checkboxes = document.querySelectorAll('input[name="car_types[]"]');
            const carSelected = Array.from(checkboxes).some(cb => cb.checked);

            const nameRegex = /^[A-Za-zÀ-ÖØ-öø-ÿ' -]+$/u;
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

            let clientErrors = [];

            if (!name) clientErrors.push('Name is required');
            else if (!nameRegex.test(name)) clientErrors.push('Name may contain only letters, spaces, apostrophes and hyphens');

            if (!phone) clientErrors.push('Phone number is required');
            else if (!/^[0-9]{10}$/.test(phone)) clientErrors.push('Phone must be 10 digits');

            if (!email) clientErrors.push('Email is required');
            else if (!emailRegex.test(email)) clientErrors.push('Please enter a valid email address');

            if (!carSelected) clientErrors.push('Please select at least one car type');

            if (clientErrors.length > 0) {
                alert(clientErrors.join('\n'));
                return;
            }

            try {
                const checkRes = await fetch('processform.php', {
                    method: 'POST',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    body: 'action=check_email&email=' + encodeURIComponent(email)
                });

                const checkJson = await checkRes.json();
                if (checkJson.exists) {
                    alert('This email has already been used');
                    return;
                }
            } catch (err) {
                console.warn('Email check failed, proceeding with submission');
            }

            try {
                const formData = new FormData(carForm);
                const res = await fetch('processform.php', {
                    method: 'POST',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    body: formData
                });

                const json = await res.json();
                if (json.success) {
                    window.location = json.redirect || 'viewsubmission.php?success=1';
                } else {
                    alert((json.errors || ['Unknown error']).join('\n'));
                }
            } catch (err) {
                carForm.submit();
            }
        });
    </script>
</body>
</html>